<?php

namespace App\Http\Controllers;

use App\Models\SystemSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\Models\Company;
use Illuminate\Support\Facades\DB;

class SystemSettingController extends Controller
{
    public function index()
    {
        $companyId = getCompanyId();

        // Retrieve system settings for the logged-in user's company
        $settings = SystemSetting::where('company_id', $companyId)->first();

        // If no settings exist, return default values
        if (!$settings) {
            $settings = [
                'currency' => 'AF',
                'language' => 'en',
                'stock_amount_status' => 20,
            ];
        }

        return response()->json($settings);
    }

    public function update(Request $request)
    {
        $request->validate([
            'currency' => 'required|string|max:5',
            'language' => 'required|string|max:10',
            'stock_amount_status' => 'required|integer',
        ]);

        $companyId = getCompanyId();

        $settings = SystemSetting::firstOrCreate(
            ['company_id' => $companyId],
            ['currency' => 'AF', 'language' => 'en',
             'stock_amount_status' => 20,
             ]
        );

        // Update system settings
        $settings->update([
            'currency' => $request->currency,
            'language' => $request->language,
            'stock_amount_status' => $request->stock_amount_status,
        ]);

        return response()->json(['message' => 'Settings updated successfully']);
    }


    //======================== Backup Manually

    public function backupCompanies()
    {
        $companyId = getCompanyId();
        $company = Company::where('id', $companyId)->first();
        $companyName = $company->name;

        $database = env('DB_DATABASE');
        $user = env('DB_USERNAME', 'default_user');
        $password = env('DB_PASSWORD', 'default_password');
    
        $timestamp = Carbon::now()->format('Y-m-d_H-i-s');
        $folderPath = "C:/backup/{$companyName}-{$companyId}/";
        $filename = "backup-{$companyId}-{$timestamp}.sql";
        $storagePath = $folderPath . $filename;
    
        if (!is_dir($folderPath)) {
            mkdir($folderPath, 0777, true);
        }
    
        // Tables to back up
        $tables = [
            'company_users', 'categories', 'company_profiles',
            'products', 'suppliers', 'purchases', 'stocks',
            'sales', 'sale_items', 
            'system_settings', 'medicine_types', 'expenses',
            'reversal_logs', 'sale_payments',
        ];
    
        // Initialize backup file
        file_put_contents($storagePath, ""); // Clear or create the backup file
    
        foreach ($tables as $table) {
            $command = "mysqldump -u {$user} --password=\"{$password}\" {$database} {$table} --where=\"company_id={$companyId}\" >> \"{$storagePath}\"";
            exec($command, $output, $resultCode);
    
            // Log::info(['result code backup:'=> $resultCode]);

            if ($resultCode !== 0) {
                return response()->json(['message' => "Backup failed for table {$table}"], 500);
            }
        }
    
        return response()->json(['message' => 'Backup completed successfully'], 200);
    }
}