<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SupplierController extends Controller
{
    /**
     * Display a listing of suppliers.
     */
    public function index(Request $request)
    {
        $companyId = getCompanyId();
        $query = Supplier::where('company_id', $companyId)
                    ->orderBy('created_at', 'desc');

        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where('name', 'like', '%' . $search . '%');
        }
        $supplier = $query->paginate(6);
        $supplier->transform(function ($item) {
            $item->hashed_id = substr(md5($item->id), 0, 8);
            return $item;
        });
        return response()->json($supplier);
    }


    public function store(Request $request)
    {
        $companyId = null;
        $companyId = getCompanyId();
        if (is_null($companyId)) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }
    
        // Step 2: Validate the incoming request
        try {
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'contact_person' => 'required|string|max:255',
                'phone_number' => 'required|string|max:15',
                'email' => 'nullable|email|max:255',
                'address' => 'required|string|max:500',
                'join_date' => 'required|date',
                'total_money' => 'required|numeric|min:0',
                'received_money' => 'required|numeric|min:0',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['error' => 'Validation failed', 'messages' => $e->errors()], 422);
        }
    
        // Step 3: Create the Supplier record
        try {
            $supplier = Supplier::create([
                'name' => $request->name,
                'contact_person' => $request->contact_person,
                'phone_number' => $request->phone_number,
                'email' => $request->email,
                'address' => $request->address,
                'join_date' => $request->join_date,
                'total_money' => $request->total_money,
                'received_money' => $request->received_money,
                'remaining_money' => $request->total_money - $request->received_money,
                'company_id' => $companyId,
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error creating supplier'], 500);
        }
        return response()->json(['message' => 'Supplier added successfully', 'supplier' => $supplier], 200);
    }
    
    

    /**
     * Display the specified supplier.
     */
    public function show($id)
    {
        $supplier = Supplier::find($id);

        if (!$supplier) {
            return response()->json(['message' => 'Supplier not found'], 404);
        }
        return response()->json($supplier, 200);
    }

    /**
     * Update the specified supplier.
     */
    public function update(Request $request, $id)
    {
        $supplier = Supplier::find($id);
        if (!$supplier) {
            return response()->json(['message' => 'Supplier not found'], 404);
        }
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'contact_person' => 'required|string|max:255',
            'phone_number' => 'required|string|max:15',
            'email' => 'nullable|email|max:255',
            'address' => 'sometimes|string|max:500',
            'join_date' => 'sometimes|date',
            'total_money' => 'sometimes|numeric|min:0',
            'received_money' => 'sometimes|numeric|min:0',
            'remaining_money' => 'sometimes|numeric|min:0',
        ]);
        $supplier->update($validatedData);
        return response()->json(['message' => 'Supplier updated successfully', 'supplier' => $supplier], 200);
    }

    /**
     * Remove the specified supplier.
     */
    public function destroy($id)
    {
        try {
            $supplier = Supplier::findOrFail($id);
            $supplier->delete();
            return response()->json(['message' => 'supp$supplier and its image deleted successfully.'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Internal Server Error'], 500);
        }
    }


    //===================

    public function getSupplierData()
    {
        $companyId = getCompanyId();
        $supplier = Supplier::where('company_id', $companyId)
                    ->orderBy('created_at', 'desc')
                    ->get();

        return response()->json($supplier);
    }

    //============== Pay Money to supplier

    public function StorePayment(Request $request) {
        try {
    
            $validated = $request->validate([
                'supplier_id' => 'required|exists:suppliers,id',
                'amount' => 'required|integer|min:1',
                'date' => 'required|date',
            ]);
    
            $amount = $validated['amount'];
            $payment = Payment::create([
                'supplier_id' => $validated['supplier_id'],
                'amount' => $validated['amount'],
                'date' => $validated['date'],
            ]);

            $supplier = Supplier::find($validated['supplier_id']);
            if($supplier){
                $supplier->increment('received_money', $amount);
            }

            return response()->json(['category' => $payment], 201);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to create category'], 500);
        }

    }

    //=================  Show Payment

    public function ShowPayment($supplierId) {
        $payments = Payment::where('supplier_id', $supplierId)->get();
        return response()->json(['payments'=>$payments]);
    }
}