<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Supplier;
use Illuminate\Http\Request;

class ProductSupplierController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Fetch all product-supplier relationships with related product and supplier data
        $products = Product::with('suppliers')->get();
        return response()->json($products);
    }

    /**
     * Show the form for creating a new resource (optional for API).
     */
    public function create()
    {
        // Optional if you're using a Vue/React frontend
        return view('product_supplier.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'supplier_id' => 'required|exists:suppliers,id',
            'unit_price' => 'required|numeric',
            'quantity' => 'required|integer',
        ]);

        // Find product and attach supplier with pivot data
        $product = Product::findOrFail($validated['product_id']);
        $product->suppliers()->attach($validated['supplier_id'], [
            'unit_price' => $validated['unit_price'],
            'quantity' => $validated['quantity'],
        ]);

        return response()->json(['message' => 'Product-Supplier association added successfully']);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $product = Product::with('suppliers')->findOrFail($id);
        return response()->json($product);
    }

    /**
     * Show the form for editing the specified resource (optional for API).
     */
    public function edit($id)
    {
        // Optional for APIs
        $product = Product::findOrFail($id);
        return view('product_supplier.edit', compact('product'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'unit_price' => 'required|numeric',
            'quantity' => 'required|integer',
        ]);

        $product = Product::findOrFail($id);

        // Update the pivot table record
        $product->suppliers()->updateExistingPivot($validated['supplier_id'], [
            'unit_price' => $validated['unit_price'],
            'quantity' => $validated['quantity'],
        ]);

        return response()->json(['message' => 'Product-Supplier association updated successfully']);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        // Example: Detach a supplier from the product
        $product = Product::findOrFail($id);
        $product->suppliers()->detach(request('supplier_id'));

        return response()->json(['message' => 'Product-Supplier association removed successfully']);
    }
}
