<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Stock;
use App\Models\SystemSetting;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;


class ProductController extends Controller
{
    /**
     * Display a listing of the products.
     */
    

public function index(Request $request)
{
    $companyId = getCompanyId();

    $query = Product::where('company_id', $companyId)
        ->with(['stock', 'medicineType'])
        ->orderBy('created_at', 'desc');

    if ($request->has('search') && !empty($request->search)) {
        $query->where('name', 'like', '%' . $request->search . '%');
    }

    if ($request->has('category_id') && $request->category_id != 'all') {
        $query->where('category_id', $request->category_id);
    }

    if ($request->has('medicine_type_id') && $request->medicine_type_id != 'all') {
        $query->where('medicine_type_id', $request->medicine_type_id);
    }

    if ($request->has('expired') && $request->expired === '1') {
        $query->whereDate('expiration_date', '<', now());
    }

    if ($request->has('active') && $request->active === '1') {
        $query->whereDate('expiration_date', '>', now());
    }

    $product = $query->paginate(7);

    $product->transform(function ($item) {
        $item->hashed_id = substr(md5($item->id), 0, 8);
        $item->stock_amount = $item->stock->quantity ?? 0;
        $quantity = optional($item->stock)->quantity ?? 0;

        $settings = SystemSetting::where('company_id', getCompanyId())->first();
        $settings_stock_amount = $settings->stock_amount_status ?? 20;

        $item->stock_status = match(true) {
            $quantity == 0 => 'Out of Stock',
            $quantity < $settings_stock_amount => 'Low Stock',
            default => 'High Stock',
        };

        return $item;
    });

    return response()->json($product);
}
    
    
    

    /**
     * Show the form for creating a new product.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created product in the database.
     */
    public function store(Request $request)
    {
        try {

            $companyId = null;
            $companyId = getCompanyId();
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'category_id' => 'required|exists:categories,id',
                'medicine_type_id' => 'required|exists:medicine_types,id',
                'product_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:1024',  // Correct validation rule
                'quantity' => 'required|integer|min:1',
                'purchase_price' => 'required|numeric|min:0',
                'sales_price' => 'required|numeric|min:0',
                'manufacturing_date' => 'required|date',
                'expiration_date' => 'nullable|date|after:manufacturing_date',
                'barcode' => 'required|unique:products,barcode',
            ]);
    
            $product_imagePath = null;
            if ($request->hasFile('product_image')) {
                $extension = $request->file('product_image')->getClientOriginalExtension();
                $fileName = 'product_' . time() . '.' . $extension;
                $product_imagePath = $request->file('product_image')->storeAs('products', $fileName, 'public');
            }

            $product = Product::create([
                'name' => $request->name,
                'category_id' => $request->category_id,
                'medicine_type_id' => $request->medicine_type_id,
                'product_image' => $product_imagePath,
                'quantity' => $request->quantity,
                'purchase_price' => $request->purchase_price,
                'sales_price' => $request->sales_price,
                'manufacturing_date' => $request->manufacturing_date,
                'expiration_date' => $request->expiration_date,
                'barcode' => $request->barcode,
                'company_id' => $companyId,
            ]);
    
            return response()->json([
                'message' => 'Product added successfully!',
                'product' => $product
            ], 201);
    
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Internal Server Error'
            ], 500);
        }
    }
    
    
    

    /**
     * Display the specified product.
     */
    public function show($id)
    {
        $product = Product::find($id);

        if (!$product) {
            return response()->json(['message' => 'product not found'], 404);
        }
        return response()->json($product, 200);
    }

    /**
     * Show the form for editing the specified product.
     */
    public function edit(Product $product)
    {
        //
    }

    /**
     * Update the specified product in the database.
     */

    public function update(Request $request, $id)
    {
        try {

            Log::info('updated product:');

            $companyId = null;
            $companyId = getCompanyId();
            $product = Product::where('company_id', $companyId)->findOrFail($id);
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'category_id' => 'required|exists:categories,id',
                'medicine_type_id' => 'required|exists:medicine_types,id',
                'product_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:1024',
                'quantity' => 'required|integer|min:1',
                'purchase_price' => 'required|numeric|min:0',
                'sales_price' => 'required|numeric|min:0',
                'manufacturing_date' => 'required|date',
                'expiration_date' => 'required|date|after:manufacturing_date',
                'barcode' => 'required',
            ]);

            $product_imagePath = null;
        if ($request->hasFile('product_image')) {
            if($product->product_image){
                Storage::disk('public')->delete($product->product_image);
            }
            
            $logoFile = $request->file('product_image');
            $extension = $logoFile->getClientOriginalExtension();
            $logoName = 'product_' . time() . '.' . $extension;
            $product_imagePath = $logoFile->storeAs('products', $logoName, 'public');
            $validated['product_image'] = $product_imagePath;
        }else{
            $validated['product_image'] = $product->product_image;
        }
            $product->update($validated);

            Log::info(['updated product:'=> $product]);

            return response()->json(['message'=>'Product updated successfully', 'product'=>$product], 200);

        }catch(\Exception $e){
            return response()->json(['error'=>'Internal server error']);
        }
    }

    /**
     * Remove the specified product from the database.
     */
    public function destroy($id)
    {
        Log::info('delete method');
        try {
            $product = Product::findOrFail($id);
            if ($product->product_image) {
                $imagePath = storage_path('app/public/' . $product->product_image);
                if (file_exists($imagePath)) {
                    unlink($imagePath);
                } else {
                    Log::warning('Product image not found: ' . $imagePath);
                }
            }
            $product->delete();
    
            return response()->json(['message' => 'Product and its image deleted successfully.'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Internal Server Error'], 500);
        }
    }
    

    /**
     * Get the count of expired products for the authenticated user's company.
     */
    public function countExpiredProducts()
    {
        try {
            $companyId = null;
            $companyId = getCompanyId();
            $expiredCount = Product::where('company_id', $companyId)
                ->whereDate('expiration_date', '<=', Carbon::now()) // Compare with the current date
                ->count();
            return response()->json(['expired_products_count' => $expiredCount], 200);
        } catch (\Exception $e) {
            Log::error('Error calculating expired products: ' . $e->getMessage());
            return response()->json(['error' => 'Internal Server Error'], 500);
        }
    }



    /**
     * Get the count of all products for the authenticated user's company.
     */
    public function countAllProducts()
    {
        try {
            $companyId = null;
            $companyId = getCompanyId();
            $productCount = Product::where('company_id', $companyId)->count();
            return response()->json(['total_products_count' => $productCount], 200);
        } catch (\Exception $e) {
            Log::error('Error calculating total products: ' . $e->getMessage());
            return response()->json(['error' => 'Internal Server Error'], 500);
        }
    }



    public function getExpiredProducts(){
        try {
            $companyId = null;
            $companyId = getCompanyId();
            $expiredProduct = Product::where('company_id', $companyId)
                ->whereDate('expiration_date', '<=', Carbon::now()) // Compare with the current date
                ->get();
            return response()->json($expiredProduct);
        } catch (\Exception $e) {
            Log::error('Error calculating expired products: ' . $e->getMessage());
            return response()->json(['error' => 'Internal Server Error'], 500);
        }
    }

    //===============================================
    
    public function getNearExpiredProducts(){
        try {
            $companyId = null;
            $companyId = getCompanyId();
            $nearExpiredProduct = Product::where('company_id', $companyId)
                ->whereBetween('expiration_date', [Carbon::now(), Carbon::now()->addDay(20)]) // Compare with the current date
                ->get()
                ->map( function ($product) {
                    $dayLeft = Carbon::now()->diffInDays(Carbon::parse($product->expiration_date));
                    $product->days_to_expiry = $dayLeft + 1 . ' ' . 'days later';
                    return $product;
                });
            return response()->json($nearExpiredProduct);
        } catch (\Exception $e) {
            Log::error('Error calculating expired products: ' . $e->getMessage());
            return response()->json(['error' => 'Internal Server Error'], 500);
        }
    }


    //=================================  Out of stock products

    public function OutOfStockProducts()
    {
        $companyId = getCompanyId();
        $products = Product::where('company_id', $companyId)
            ->whereDoesntHave('stock')
            ->orWhereHas('stock', function ($query) {
                $query->where('quantity', 0);
            })
            ->with('stock')
            ->orderBy('created_at', 'desc')
            ->get();
    
        return response()->json($products);
    }

}
