<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Company;
use App\Models\CompanyUser;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Inertia\Inertia;
use Inertia\Response;
use Carbon\Carbon;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): Response
    {
        $user = auth()->user();
        return Inertia::render('Auth/Register',[
            'userRole' => $user->role,
        ]);
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $currentUser = Auth::user();
        
        // Validation
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email|unique:companies,email',
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);
    
        // Check for the current user role
        if ($currentUser && $currentUser->role === 'system_admin') {
            // If system admin is registering a company, store in `companies` table
            $company = Company::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'user_id' => $currentUser->id, // System admin ID as foreign key
                // Add other company-specific fields if needed
            ]);
    
            return redirect()->route('dashboard')->with('message', 'Company registered successfully.');
            
        } 

        elseif ($currentUser && $currentUser->role === 'admin') {
            // If a company admin is registering a user, store in `company_users` table
            $companyUser = CompanyUser::create([
                'name' => $request->name, // Ensure `name` field exists in `company_users` table
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'role' => $request->role,
                'company_id' => $currentUser->id,
                'register_date' => Carbon::now(),
            ]);
    
            return redirect()->route('userdashboard')->with('message', 'User registered under company successfully.');
        }
        
        // Handle the case where registration is not allowed
        return redirect()->route('/')->withErrors(['message' => 'Registration not allowed.']); // Use a named route for better clarity
    }
    
    
    
}
