<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): Response
    {
        return Inertia::render('Auth/Login', [
            'canResetPassword' => true, // Indicate whether the reset password option is available
            'resetPasswordUrl' => route('password.request'), // Pass the URL as a separate prop
            'status' => session('status'),
            'loginError' => session('loginError'),
        ]);
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        $credentials = $request->only('email', 'password');

        // Check system admin (web guard)
        if (Auth::guard('web')->attempt($credentials)) {
            $request->session()->regenerate();
            return redirect()->intended('/dashboard');
        }

        // Check company admin (admin guard)
        if ($user = \App\Models\Company::where('email', $credentials['email'])->first()) {
            if ($user->is_locked) {
                return redirect()->route('login')->with('loginError', 'Your account is locked by System Admin.');
            }

            if (Auth::guard('admin')->attempt($credentials)) {
                $request->session()->regenerate();
                return redirect()->intended('/userdashboard');
            }
        }

        // Check company user (user guard)
        if ($user = \App\Models\CompanyUser::where('email', $credentials['email'])->first()) {
            if ($user->is_locked) {
                return redirect()->route('login')->with('loginError', 'Your account is locked by Manager.');
            }

            if (Auth::guard('user')->attempt($credentials)) {
                $request->session()->regenerate();
                return redirect()->intended('/userdashboard');
            }
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ]);
    }
    
    

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        // Log out from all guards; adjust if you prefer only the current guard logout.
        Auth::guard('web')->logout();
        Auth::guard('admin')->logout();
        Auth::guard('user')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }
}
